#####################################################################
## This program is distributed in the hope that it will be useful, ##
## but WITHOUT ANY WARRANTY; without even the implied warranty of  ##
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the    ##
## GNU General Public License for more details.                    ##
#####################################################################

#-------------------------------------------------------------------------------
# gtoxReport: Generate a report
#-------------------------------------------------------------------------------

#' @title Generate a report
#'
#' @description
#' \code{gtoxReport} generates a report.
#'
#' @param type The type of report to generate
#' @param asid The assay source/study ID
#' @param params Named list containing report type-specific parameters, see
#' details
#' @param odir The output directory
#' @param report_author The author for the report
#' @param report_title The title for the report
#' @param sumfile Path to a text file that inserts into the report
#' @param keep.intermediates TRUE/FALSE, keep intermediate files when TRUE
#'
#' @details
#' 'type' can have three values, "all," "compare," or "qc." Each report
#' contains slightly different elements, but in general:
#' \itemize{
#' \item "all" -- summarizes the results for all or some compounds
#'   \itemize{
#'   \item "chids" -- (optional) a vector of chid values to report, rather than
#'   all available compounds
#'   }
#' \item "compare" -- compares the results for two compounds
#'   \itemize{
#'   \item "c1" -- (required) the chid for the first compound to compare
#'   \item "c2" -- (required) the chid for the second compound to compare
#'   }
#' \item "qc" -- summarizes low-level data for quality control purposes
#'   \itemize{
#'   \item "aids" -- (optional) a vector of aid values to report, rather than
#'   all available assays
#'   }
#' }
#'
#' The required list elements vary depending on the type of report, and are
#' described under the report descriptions above.
#'
#' 'sumfile' allows the user to inject a Tex file into the report. The
#' file contents will be inserted into the Study Overview section, immediately
#' after the autogenerated text. Technically, 'sumfile' is brewed, so 'sumfile'
#' can make use of brew and Sweave syntax, and all data loaded for the report.
#'
#' @examples
#' ## Generate full analysis report
#' 
#' \dontrun{
#' ## Generate report
#' gtoxReport(type = "all", asid = 1L, report_author = "author", 
#' report_title = "Processing report")
#' }
#' 
#' @return None
#'
#' @import data.table
#' @importFrom tools texi2pdf
#' @importFrom utils Sweave
#' @importFrom brew brew
#' @importFrom xtable xtable sanitize
#' @importFrom RColorBrewer brewer.pal
#' @export

gtoxReport <- function(type, asid, params=NULL, odir=getwd(),
    report_author,
    report_title="Report", sumfile=NULL,
    keep.intermediates=FALSE) {

    ## replace any LaTeX spec chars
    report_title <- gsub("[[:punct:]]", " ", report_title)
    report_author <- gsub("[[:punct:]]", " ", report_author)

    ## Check type
    if (!type %in% c("compare", "qc", "all", "test")) {
        stop("Invalid 'type' input; must be 'compare,' 'qc,' or 'all'")
    }

    ## Check params
    if (type == "compare") {
        if (is.null(params$c1) | is.null(params$c2)) {
            stop("'params' must include 'c1' and 'c2' for the compare report.")
        }
        if (length(params$c1) != 1 | !is.numeric(params$c1)) {
            stop("Invalid value for 'params$c1'")
        }
        if (length(params$c2) != 1 | !is.numeric(params$c2)) {
            stop("Invalid value for 'params$c2'")
        }
        c1 <- params$c1
        c2 <- params$c2
    }
    if (type == "all") {
        if (!is.null(params$chids) & !is.numeric(params$chids)) {
            stop("Invalid value for 'params$chids'")
        }
    }
    if (type == "qc") {
        if (!is.null(params$aids) & !is.numeric(params$aids)) {
            stop("Invalid value for 'params$aids'")
        }
    }

    ## Create the file name based upon report type and input data
    labl <- switch(
        type,
        compare=paste0("Compare_", c1, "_", c2),
        qc="QC",
        all="AllCompounds",
        test="Test"
    )

    fname <- paste0(
        format(Sys.Date(), "%y%m%d_"), "HCS_", labl, "_ASID", asid
    )

    ## Path to the template file
    tmp_fp <- file.path(
        system.file(package="GladiaTOX"),
        "report_templates",
        paste0(type, ".Rnw")
    )

    ## Expand path to sumfile
    if (!is.null(sumfile)) sumfile <- normalizePath(sumfile)

    ## Create a temporary directory for the intermediate files, change directory
    tdir <- tempfile(pattern="tmp", tmpdir=odir)
    dir.create(tdir)
    owd <- getwd()
    setwd(tdir)

    ## Delete temporary files and return to original directory on function exit
    on.exit(setwd(owd))
    if (!keep.intermediates) on.exit(unlink(tdir, recursive=TRUE), add=TRUE)

    ## brew, weave, and create pdf file
    brew(file=tmp_fp, output=paste0(fname, ".Rnw"))
    Sweave(paste0(fname, ".Rnw"))
    texi2pdf(paste0(fname, ".tex"))
    pdf_file <- paste0(fname, ".pdf")

    ## Move final report to the output directory (odir)
    file.rename(from=pdf_file, to=file.path(odir, pdf_file))

}
