% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/taxa_barplot.R
\name{taxa_barplot}
\alias{taxa_barplot}
\title{Generate a Taxa Barplot with Relative or Absolute Abundance}
\source{
Built on public functions from phyloseq and ggplot2 for data transformation and plotting.
}
\usage{
taxa_barplot(
  physeq,
  target_glom = "Genus",
  custom_tax_names = NULL,
  normalize = TRUE,
  treatment_variable = "Treatment",
  abundance_type = "relative",
  x_angle = 25,
  fill_variable = target_glom,
  facet_variable = NULL,
  top_n_taxa = 20,
  palette = color_palette$MG,
  legend_size = 11,
  legend_columns = 1,
  x_scale = "free",
  xlab = NULL
)
}
\arguments{
\item{physeq}{A \code{phyloseq} or \code{TreeSummarizedExperiment} object containing microbiome data.}

\item{target_glom}{A character string specifying the taxonomic rank to plot (e.g., "Genus").}

\item{custom_tax_names}{A character vector specifying custom taxonomic names for the levels. Default is \code{NULL}.}

\item{normalize}{A logical value indicating whether to normalize sample counts to relative abundances. Default is \code{TRUE}.}

\item{treatment_variable}{A character string specifying the treatment variable for the x-axis (e.g., "Treatment"). Default is \code{"Treatment"}.}

\item{abundance_type}{A character string specifying whether to plot \code{"relative"} or \code{"absolute"} abundance. Default is \code{"relative"}.}

\item{x_angle}{A numeric value specifying the angle of x-axis text labels. Default is \code{25}.}

\item{fill_variable}{A character string specifying the variable to use for filling bar colors. Default is the same as \code{target_glom}.}

\item{facet_variable}{A character string specifying the variable to use for faceting the plot. Default is \code{NULL} (no faceting).}

\item{top_n_taxa}{A numeric value specifying the number of top taxa to include in the plot. Default is \code{20}.}

\item{palette}{A character vector of color codes or a function generating such a palette. Default is \code{color_palette$MG}.}

\item{legend_size}{A numeric value specifying the size of the legend text. Default is \code{11}.}

\item{legend_columns}{A numeric value specifying the number of columns for the legend. Default is \code{1}.}

\item{x_scale}{A character string specifying the x-axis scale in facets. Options are \code{"free"} or \code{"fixed"}. Default is \code{"free"}.}

\item{xlab}{A character string specifying the x-axis label. Default is \code{NULL} (no label).}
}
\value{
A list containing the following components:
\describe{
\item{barplot}{A \code{ggplot2} object representing the taxa barplot.}
\item{taxa_data}{A \code{phyloseq} object containing the top taxa and aggregated "Others" taxa.}
}
}
\description{
This function creates a bar plot of relative or absolute abundances of the top \code{n} taxa
(OTUs/ASVs) at a specified taxonomic rank. Non-top taxa are aggregated into an "Others" category,
which is always displayed at the top for clarity. The function supports normalization, faceting,
and customization of plot aesthetics. The plot is built using \code{ggplot2}, and the taxa are ordered
with "Others" appearing first.
}
\examples{
# Example 1: Relative abundance barplot (subset to Frog samples for speed)
data("physeq_16SOTU", package = "DspikeIn")

# Subset to only 'Frog' samples
physeq_frog <- phyloseq::subset_samples(physeq_16SOTU, Animal.type == "Frog")

# Remove taxa with zero abundance
physeq_frog <- phyloseq::prune_taxa(phyloseq::taxa_sums(physeq_frog) > 0, physeq_frog)

# Plot relative abundance for the top 5 genera
bp_rel <- taxa_barplot(
  physeq = physeq_frog,
  target_glom = "Genus",
  fill_variable = "Family",
  treatment_variable = "Diet",
  abundance_type = "relative",
  top_n_taxa = 5,
  legend_size = 9,
  x_scale = "fixed",
  legend_columns = 1,
  x_angle = 15,
  palette = DspikeIn::color_palette$MG
)
print(bp_rel$barplot)


# Example 2: Absolute abundance barplot (ITS data converted to TSE; Frog subset)
data("physeq_ITSOTU", package = "DspikeIn")

# Convert the phyloseq object to TreeSummarizedExperiment
tse_ITSOTU <- convert_phyloseq_to_tse(physeq_ITSOTU)
tse_frog <- tse_ITSOTU[, SummarizedExperiment::colData(tse_ITSOTU)$Animal.type == "Frog"]
tse_frog <- tse_frog[rowSums(SummarizedExperiment::assay(tse_frog)) > 0, ]

# Plot absolute abundance for top 5 Families
bp_abs <- taxa_barplot(
  physeq = tse_frog,
  target_glom = "Family",
  treatment_variable = "Diet",
  fill_variable = "Family",
  abundance_type = "absolute",
  top_n_taxa = 5,
  x_angle = 15,
  legend_size = 9,
  legend_columns = 1,
  x_scale = "fixed",
  palette = DspikeIn::color_palette$cool_MG
)
print(bp_abs$barplot)

}
