% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computePMDs.R
\name{computePMDs}
\alias{computePMDs}
\title{Compute PMDs}
\usage{
computePMDs(
  methylationData,
  regions = NULL,
  context = "CG",
  method = "noise_filter",
  windowSize = 100,
  kernelFunction = "triangular",
  lambda = 0.5,
  binSize = 100,
  minCytosinesCount = 4,
  minMethylation = 0.4,
  maxMethylation = 0.6,
  minGap = 200,
  minSize = 50,
  minReadsPerCytosine = 4,
  parallel = FALSE,
  BPPARAM = NULL,
  cores = NULL
)
}
\arguments{
\item{methylationData}{the methylation data in condition
(see \code{\link{ontSampleGRangesList}}).}

\item{regions}{a \code{\link[GenomicRanges]{GRanges}} object with the regions where to 
compute the PMDs. If \code{NULL}, the PMDs are computed genome-wide.}

\item{context}{the context in which the PMDs are computed (\code{"CG"}, 
\code{"CHG"} or \code{"CHH"}).}

\item{method}{Character string specifying the algorithm for PMD detection. 
If \code{"noise_filter"}, a sliding window of size \code{windowSize} is applied 
with the specified \code{kernelFunction} (and \code{lambda} for a Gaussian kernel) 
to smooth methylation counts before calling and merging PMDs. 
If \code{"neighbourhood"}, individual partially methylated cytosines are 
identified first and then merged into PMDs. 
If \code{"bins"}, the genome is partitioned into fixed bins of size \code{binSize}, 
partially methylation is sorted per bin, and significant bins are merged.}

\item{windowSize}{the size of the triangle base measured in nucleotides. 
This parameter is required only if the selected method is 
\code{"noise_filter"}.}

\item{kernelFunction}{a \code{character} indicating which kernel function to 
be used. Can be one of \code{"uniform"}, \code{"triangular"}, 
\code{"gaussian"} or \code{"epanechnicov"}. This is required only if the 
selected method is \code{"noise_filter"}.}

\item{lambda}{numeric value required for the Gaussian filter 
(\code{K(x) = exp(-lambda*x^2)}). This is required only if the selected 
method is \code{"noise_filter"} and the selected kernel function is 
\code{"gaussian"}.}

\item{binSize}{the size of the tiling bins in nucleotides. This parameter is 
required only if the selected method is \code{"bins"}.}

\item{minCytosinesCount}{PMDs with less cytosines in the specified context 
than \code{minCytosinesCount} will be discarded.}

\item{minMethylation}{Numeric [0,1]; minimum mean methylation proportion.}

\item{maxMethylation}{Numeric [0,1]; maximum mean methylation proportion.}

\item{minGap}{PMDs separated by a gap of at least \code{minGap} are not 
merged. Note that only PMDs where the change in methylation is in the same 
direction are joined.}

\item{minSize}{PMDs with a size smaller than \code{minSize} are discarded.}

\item{minReadsPerCytosine}{PMDs with the average number of reads lower than 
\code{minReadsPerCytosine} are discarded.}

\item{parallel}{Logical; run in parallel if \code{TRUE}.}

\item{BPPARAM}{A \code{BiocParallelParam} object controlling parallel execution.
This value will automatically set when parallel is \code{TRUE}, also able to set as manually.}

\item{cores}{Integer number of workers (must not exceed BPPARAM$workers).
This value will automatically set as the maximum number of system workers,
also able to set as manually.}
}
\value{
the PMDs stored as a \code{\link[GenomicRanges]{GRanges}} object with the following 
metadata columns:
\describe{
 \item{context}{the context in which the PMDs was computed (\code{"CG"}, 
 \code{"CHG"} or \code{"CHH"}).}
 \item{sumReadsM}{the number of methylated reads.}
 \item{sumReadsN}{the total number of reads.} 
 \item{proportion}{the proportion methylated reads filtered between
 \code{minMethylation} and \code{maxMethylation}.} 
 \item{cytosinesCount}{the number of cytosines in the PMDs.} 
}
}
\description{
This function computes the partially methylated domains between pre-set
min and max proportion values.
}
\examples{
# load the ONT methylation data 
data(ontSampleGRangesList)

# the regions where to compute the PMDs
chr1_ranges <- GRanges(seqnames = Rle("chr1"), ranges = IRanges(1E6+5E5,2E6))

# compute the PMDs in CG context with noise_filter method
PMDsNoiseFilterCG <- computePMDs(ontSampleGRangesList[["GM18501"]],
                                 regions = chr1_ranges,
                                 context = "CG",
                                 windowSize = 100,
                                 method = "noise_filter",
                                 kernelFunction = "triangular", 
                                 lambda = 0.5,
                                 minCytosinesCount = 4, 
                                 minMethylation = 0.4, 
                                 maxMethylation = 0.6, 
                                 minGap = 200, 
                                 minSize = 50, 
                                 minReadsPerCytosine = 4, 
                                 cores = 1,
                                 parallel = FALSE)
 
\dontrun{                                
# compute the PMDs in CG context with neighbourhood method
PMDsNeighbourhoodCG <- computePMDs(ontSampleGRangesList[["GM18501"]],
                                   regions = chr1_ranges,
                                   context = "CG",
                                   method = "neighbourhood"
                                   minCytosinesCount = 4, 
                                   minMethylation = 0.4, 
                                   maxMethylation = 0.6, 
                                   minGap = 200, 
                                   minSize = 50, 
                                   minReadsPerCytosine = 4, 
                                   cores = 1,
                                   parallel = FALSE)
                                   
# compute the PMDs in CG context with bins method
PMDsBinsCG <- computePMDs(ontSampleGRangesList[["GM18501"]],
                          regions = chr1_ranges,
                          context = "CG",
                          method = "bins",
                          binSize = 100,
                          minCytosinesCount = 4, 
                          minMethylation = 0.4, 
                          maxMethylation = 0.6, 
                          minGap = 200, 
                          minSize = 50, 
                          minReadsPerCytosine = 4, 
                          cores = 1,
                          parallel = FALSE)
}
}
\seealso{
\code{\link{readONTbam}}, \code{\link{filterPMDs}}, \code{\link{mergePMDsIteratively}}, 
\code{\link{analyseReadsInsideRegionsForConditionPMD}} and 
\code{\link{PMDsNoiseFilterCG}}
}
\author{
Nicolae Radu Zabet, Jonathan Michael Foonlan Tsang and Young Jun Kim
}
