% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CoreMethods.R
\name{SummariseCellClusterProbability}
\alias{SummariseCellClusterProbability}
\alias{SummariseCellClusterProbability.SingleCellExperiment}
\alias{SummariseCellClusterProbability,SingleCellExperiment-method}
\title{Summarise ICP cell cluster probability}
\usage{
SummariseCellClusterProbability.SingleCellExperiment(
  object,
  icp.run,
  icp.round,
  funs,
  scale.funs,
  save.in.sce
)

\S4method{SummariseCellClusterProbability}{SingleCellExperiment}(
  object,
  icp.run = NULL,
  icp.round = NULL,
  funs = c("mean", "median"),
  scale.funs = TRUE,
  save.in.sce = TRUE
)
}
\arguments{
\item{object}{An object of \code{SingleCellExperiment} class with ICP cell
cluster probability tables saved in \code{metadata(object)$coralysis$joint.probability}.
After running \code{RunParallelDivisiveICP}.}

\item{icp.run}{ICP run(s) to retrieve from \code{metadata(object)$coralysis$joint.probability}.
By default \code{NULL}, i.e., all are retrieved. Specify a numeric vector to
retrieve a specific set of tables.}

\item{icp.round}{ICP round(s) to retrieve from \code{metadata(object)$coralysis$joint.probability}.
By default \code{NULL}, i.e., all are retrieved.}

\item{funs}{Functions to summarise ICP cell cluster probability: \code{"mean"}
and/or \code{"median"}. By default \code{c("mean", "median")}, i.e, both mean
and median are calculated. Set to \code{NULL} to not estimate any.}

\item{scale.funs}{Scale in the range 0-1 the summarised probability obtained with
\code{funs}. By default \code{TRUE}, i.e., summarised probability will be scaled
in the 0-1 range.}

\item{save.in.sce}{Save the data frame into the cell metadata from the
\code{SingleCellExperiment} object or return the data frame. By default \code{TRUE},
i.e., the summary of probabilities retrieved is save in the SCE object in
\code{colData(object)}.}
}
\value{
A data frame or a SingleCellExperiment object with ICP cell cluster probability summarised.
}
\description{
Summarise ICP cell cluster probability table(s)
}
\examples{
# Import package
suppressPackageStartupMessages(library("SingleCellExperiment"))

# Create toy SCE data
batches <- c("b1", "b2")
set.seed(239)
batch <- sample(x = batches, size = nrow(iris), replace = TRUE)
sce <- SingleCellExperiment(
    assays = list(logcounts = t(iris[, 1:4])),
    colData = DataFrame(
        "Species" = iris$Species,
        "Batch" = batch
    )
)
colnames(sce) <- paste0("samp", 1:ncol(sce))

# Prepare SCE object for analysis
sce <- PrepareData(sce)

# Multi-level integration (just for highlighting purposes; use default parameters)
set.seed(123)
sce <- RunParallelDivisiveICP(
    object = sce, batch.label = "Batch",
    k = 2, L = 25, C = 1, train.k.nn = 10,
    train.k.nn.prop = NULL, use.cluster.seed = FALSE,
    build.train.set = FALSE, ari.cutoff = 0.1,
    threads = 2, RNGseed = 1024
)

# Integrated PCA
set.seed(125) # to ensure reproducibility for the default 'irlba' method
sce <- RunPCA(object = sce, assay.name = "joint.probability", p = 10)

# Summarise cluster probability
sce <- SummariseCellClusterProbability(
    object = sce, icp.round = 1,
    save.in.sce = TRUE
) # saved in 'colData'

# Plot the clustering result for ICP run no. 3
PlotDimRed(object = sce, color.by = "icp_run_round_3_1_clusters")

# Plot Coralysis mean cell cluster probabilities
PlotExpression(
    object = sce, color.by = "mean_probs",
    color.scale = "viridis"
)

}
\keyword{Summarise}
\keyword{cell}
\keyword{cluster}
\keyword{probability}
