############################################################           
##  R U N S
############################################################           
## A run is associated with an individual flow cell and contains
## metrics and reads generated by the instrument on which the
## flow cell was sequenced


setClass("runItem", contains = "Item",
         representation = representation(
           HrefFiles = "character", # The location of the run's files in the API
           HrefSamples = "character", # The location of the run's samples in the API
           UserUploadedBy = "list", # Information about the user who uploaded the run
           DateUploadCompleted = "character", # The date when the upload was completed
           DateUploadStarted = "character", # The date when the upload was started
           ExperimentName = "character")) # The name of the experiment, set by the user who uploads the run

## The Items is a list of runItem objects
## We'll have to implement a validator for this
setClass("runCollection", contains = "Collection")


## The main Runs object - for now contains one runItem and an AppAuth instance
## At a later point we should replace runItem with a runCollection or a 'runList' to allow for vectorized operations
setClass("Runs", contains = "Response",
         representation = representation(
           data = "runItem"))

## Top level object - the metadata
setClass("RunsSummary", contains = "Response",
         representation = representation(
           data = "runCollection"))


############################################################           
## Methods / Constructors
############################################################           

## We need to find a better way to instantiate the object ...
runItem <- function(...) ItemFromJList("runItem", list(...))

## for runCollection !!!!
runCollection <- function(...) {
  CollectionFromJList("runCollection", items = list(...))
}


##############################
## Selecting Runs - returns a named R list of Runs objects

## Trivial constructor
setMethod("Runs", "missing", function() new("Runs"))

## Constructor from AppAuth
setMethod("Runs", "AppAuth",
          function(x, id, simplify = FALSE) {
            ## if 'id' is missing, first call listRuns to get all possible Ids. 
            if(missing(id))
              id <- Id(listRuns(x))  # list all Runs

            .queryResource(x = new("Runs", auth = x), "runs", id, simplify)
          })

## Selects all the runs listed in the RunsSummary instance
setMethod("Runs", "RunsSummary",
          function(x, simplify = FALSE) {
            .queryResource(x = new("Runs", auth = x@auth), "runs", Id(x), simplify)
          })


## count from AppAuth
setMethod("countRuns", "AppAuth",
          function(x) {
            res <- x$doGET(resource = "users/current/runs", Limit = 0)
            if(is.null(res))
              return(NA_integer_)

            return(as.integer(res$TotalCount))
          })

## count from any Response instance
setMethod("countRuns", "Response", function(x) countRuns(x@auth))


## List from AppAuth
setMethod("listRuns", "AppAuth",
          function(x, ...) {
            res <- x$doGET(resource = "users/current/runs", ...)
            if(is.null(res))
              return(NULL)
            
            if(!"Items" %in% names(res))
              stop("Response is not a proper JSON representation of a collection. 'Items' missing!")
            ## each entry in Items must be a runItem instance
            res$Items <- lapply(res$Items, function(l) ItemFromJList("runItem", l))

            new("RunsSummary", data = CollectionFromJList("runCollection", l = res), auth = x)
          })

## List from any Response instance
setMethod("listRuns", "Response", function(x, ...) listRuns(x@auth, ...))
