% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster.R
\name{clusterBanksy}
\alias{clusterBanksy}
\title{Perform clustering in BANKSY's neighborhood-augmented feature space.}
\usage{
clusterBanksy(
  se,
  use_agf = FALSE,
  lambda = 0.2,
  use_pcs = TRUE,
  npcs = 20L,
  dimred = NULL,
  ndims = NULL,
  assay_name = NULL,
  group = NULL,
  algo = c("leiden", "louvain", "kmeans", "mclust"),
  k_neighbors = 50,
  resolution = 1,
  leiden.iter = -1,
  kmeans.centers = 5,
  mclust.G = 5,
  M = NULL,
  seed = NULL,
  ...
)
}
\arguments{
\item{se}{A \code{SpatialExperiment},
\code{SingleCellExperiment} or \code{SummarizedExperiment}
  object with \code{computeBanksy} ran.}

\item{use_agf}{A logical vector specifying whether to use the AGF for
clustering.}

\item{lambda}{A numeric vector in \eqn{\in [0,1]} specifying a spatial
weighting parameter. Larger values (e.g. 0.8) incorporate more spatial
neighborhood and find spatial domains, while smaller values (e.g. 0.2)
perform spatial cell-typing.}

\item{use_pcs}{A logical scalar specifying whether to cluster on PCs. If
FALSE, runs on the BANKSY matrix.}

\item{npcs}{An integer scalar specifying the number of principal components
to use if \code{use_pcs} is TRUE.}

\item{dimred}{A string scalar specifying the name of an existing
dimensionality reduction result to use. Will overwrite \code{use_pcs} if
supplied.}

\item{ndims}{An integer scalar specifying the number of dimensions to use if
\code{dimred} is supplied.}

\item{assay_name}{A string scalar specifying the name of the assay used in
\code{computeBanksy}.}

\item{group}{A string scalar specifying a grouping variable for samples in
\code{se}. This is used to scale the samples in each group separately.}

\item{algo}{A string scalar specifying the clustering algorithm to use; one
of leiden, louvain, mclust, kmeans.}

\item{k_neighbors}{An integer vector specifying number of neighbors for
constructing sNN (for louvain / leiden).}

\item{resolution}{A numeric vector specifying resolution used for clustering
(louvain / leiden).}

\item{leiden.iter}{An integer scalar specifying the number of leiden
iterations. For running till convergence, set to -1 (leiden).}

\item{kmeans.centers}{An integer vector specifying the number of kmeans 
clusters (kmeans).}

\item{mclust.G}{An integer vector specifying the number of mixture 
components (Mclust).}

\item{M}{Advanced usage. An integer vector specifying the highest azimuthal
Fourier harmonic to cluster with. If specified, overwrites the
\code{use_agf} argument.}

\item{seed}{Random seed for clustering. If not specified, no seed is set.}

\item{...}{to pass to methods}
}
\value{
A SpatialExperiment / SingleCellExperiment / SummarizedExperiment
  object with cluster labels in \code{colData(se)}.
}
\description{
Perform clustering in BANKSY's neighborhood-augmented feature space.
}
\details{
This function performs clustering on the principal components computed on 
the BANKSY matrix, i.e., the BANKSY embedding. The PCA corresponding to the 
parameters \code{use_agf} and \code{lambda} must have been computed with  
\link[Banksy]{runBanksyPCA}. Clustering may also be performed directly on the 
BANKSY matrix with \code{use_pcs} set to \code{FALSE} (this is not 
recommended). 

Four clustering algorithms are implemented. 
\itemize{
 \item{leiden: Leiden graph-based clustering. The arguments 
 \code{k_neighbors} and \code{resolution} should be specified.}
 \item{louvain: Louvain graph-based clustering. The arguments 
 \code{k_neighbors} and \code{resolution} should be specified.}
 \item{kmeans: kmeans clustering. The argument \code{kmeans.centers} should 
 be specified.}
 \item{mclust: Gaussian mixture model-based clustering. The argument 
 \code{mclust.G} should be specified.}
}

By default, no seed is set for clustering. If a seed is specified, the same
seed is used for clustering across the input parameters.
}
\examples{
data(rings)
spe <- computeBanksy(rings, assay_name = "counts", M = 1, k_geom = c(15, 30))
spe <- runBanksyPCA(spe, M = 1, lambda = c(0, 0.2), npcs = 20)
spe <- clusterBanksy(spe, M = 1, lambda = c(0, 0.2), resolution = 1)

}
