\name{makeStandardOrgDbs}

\alias{makeStandardOrgDbs}
\alias{makeStandardOrgDbsToAHM}
\alias{makeStandardTxDbs}
\alias{makeStandardTxDbsToAHM}
\alias{makeNonStandardOrgDbs}
\alias{makeNCBIToOrgDbsToAHM}

\title{Functions to add OrgDb and TxDb sqlite files to AnnotationHub}

\description{Add OrgDb and TxDb sqlite files to AnnotationHub}

\usage{
makeStandardOrgDbsToAHM(currentMetadata, justRunUnitTest = FALSE, 
                        BiocVersion = BiocManager::version(), 
                        downloadOrgDbs = TRUE)

makeStandardTxDbsToAHM(currentMetadata, justRunUnitTest = FALSE, 
                       BiocVersion = BiocManager::version(), TxDbs)

makeNCBIToOrgDbsToAHM(currentMetadata, justRunUnitTest = FALSE,
                      BiocVersion = BiocManager::version(), 
                      baseUrl = "ftp://ftp.ncbi.nlm.nih.gov/gene/DATA/")
}

\arguments{
  \item{currentMetadata}{
    Historically was intended to be a list of metadata to filter, i.e.,
    records that do not need to be processed again. In some recipes this
    is used as a way to pass additional arguments. Need to remove or 
    make consistent.
  }
  \item{baseUrl}{
    A \code{character()}. The file location.
  } 
  \item{justRunUnitTest}{
    A \code{logical}. When TRUE, a small number of records (usually <= 5) are
    processed instead of all.
  }
  \item{BiocVersion}{
    A \code{character(1)}. The resource will be available for Bioconductor 
    versions greater than and equal to this version. Default is
    BiocManaer::version().
  }
  \item{TxDbs}{
    Character vector of the \code{TxDb} names; generally includes
    \code{TxDbs} that were new or updated for the current release.
  }
  \item{downloadOrgDbs}{
    A \code{logical}. Indicates if all \code{OrgDb} packages in the 
    Bioconductor repo should be downloaded and installed. This should be 
    \code{TRUE} the first time the recipe is run and can be \code{FALSE} for
    subsequent runs when testing.
  }
}

\details{
  \code{makeStandardOrgDbsToAHM} and \code{makeStandardTxDbsToAHM} extracts
  the sqlite files from the existing \code{OrgDb} and \code{TxDb} packages
  in the Bioconductor repositories and generate associated metadata.

  \code{makeNCBIToOrgDbsToAHM} creates sqlite files and metadata for 1000
  organisms with the \code{makeOrgPackageFromNCBI} function. These
  organisms are less 'main stream' than those hosted in the Bioconductor
  repository (\code{makeStandardOrgDbsToAHM}) and the databases are less 
  comprehensive because data only come from one source, NCBI.
}

\value{
  List of \code{AnnotationHubMetadata} objects. 
}

\author{Bioconductor Core Team}

\seealso{
  \itemize{
    \item \link{updateResources}
    \item \link{AnnotationHubMetadata}
  }
}

\examples{
\dontrun{
## In Bioconductor 3.5, one new TxDb was added and 4 active
## tracks were updated. This piece of code shows how to add these 5
## packages to AnnotationHub.

## Step I: generate metadata 
##
## Generate the metadata with the low-level helper for inspection.
TxDbs <- c("TxDb.Ggallus.UCSC.galGal5.refGene",
           "TxDb.Celegans.UCSC.ce11.refGene",
           "TxDb.Rnorvegicus.UCSC.rn5.refGene",
           "TxDb.Dmelanogaster.UCSC.dm6.ensGene",
           "TxDb.Rnorvegicus.UCSC.rn6.refGene")
meta <- makeStandardTxDbsToAHM(currentMetadata=list(AnnotationHubRoot="TxDbs"), 
                               justRunUnitTest=FALSE, 
                               TxDbs = TxDbs)

## Once the low-level helper runs with no errors, try generating the
## metadata with the high-level wrapper updateResources(). Setting
## metadataOnly=TRUE will generate metadata only and not push resources
## to data bucket. insert=FALSE prevents the metadata from being inserted in the
## database.
##
## The metadata generated by updateResources() will be the same as that
## generated by makeStandardTxDbsToAHM(). Both should be a list the same
## length as the number of TxDbs specified.
meta <- updateResources("TxDbs",
                        preparerClasses="TxDbFromPkgsImportPreparer",
                        metadataOnly=TRUE, insert = FALSE,
                        justRunUnitTest=FALSE, TxDbs = TxDbs)

INFO [2017-04-11 09:12:09] Preparer Class: TxDbFromPkgsImportPreparer
complete!
> length(meta)
[1] 5

## Step II: push resources to Azure 
##
## If the metadata looks correct we are ready to push resources to Azure.
## Set metadataOnly=FALSE but keep insert=FALSE.

## export an environment variable with a core generated SAS URL for
##   upload example:
##  export AZURE_SAS_URL='https://bioconductorhubs.blob.core.windows.net/staginghub?sp=racwl&st=2022-02-08T15:57:00Z&se=2022-02-22T23:57:00Z&spr=https&sv=2020-08-04&sr=c&sig=fBtPzgrw1Akzlz%2Fwkne%2BQrxOKOdCzP1%2Fk5S%2FHk1LguE%3D'

meta <- updateResources("TxDbs",
                        BiocVersion="3.5",
                        preparerClasses="TxDbFromPkgsImportPreparer",
                        metadataOnly=FALSE, insert = FALSE,
                        justRunUnitTest=FALSE, TxDbs = TxDbs)

## Step III: insert metadata in AnnotationHub production database
##
## Inserting the metadata in the database is usually done as a separte step
## and with the help of the AnnotationHub docker.
## Set metadataOnly=TRUE and insert=TRUE.
meta <- updateResources("TxDbs",
                        BiocVersion="3.5",
                        preparerClasses="TxDbFromPkgsImportPreparer",
                        metadataOnly=FALSE, insert = FALSE,
                        justRunUnitTest=FALSE, TxDbs = TxDbs)

}
}

\keyword{methods}
